import os
import json
import urllib3
import logging
from formsmarts_api import WebhookAuthenticator, FormEntry, AuthenticationError

logger = logging.getLogger()
logger.setLevel(logging.INFO)

class TeamsSender:
    def __init__(self, event):
        self._event = event
        self._au = WebhookAuthenticator(os.environ['FORMSMARTS_WEBHOOK_KEY'])
        self._entry = None

    def create_adaptive_card(self):
        """
        Creates a Microsoft Teams Adaptive Card payload.
        """
        entry = self._entry
        form_title = entry.form_name

        # 1. Build the list of facts (key-value pairs) for the card
        # Customize these lookups based on your specific form fields
        facts = [
            # Get the main message body (adjust index as needed)
            {"title": "Message", "value": entry.fields[5].value},

            # Lookup fields by type
            {"title": "Client Phone", "value": entry.fields_by_type('phone')[0].value},
            {"title": "Client Email", "value": entry.fields_by_type('email')[0].value},

            # Lookup field by name (Used for Priority check later as well)
            {"title": "Priority Level", "value": entry.fields_by_name('Priority')[0].value},
            {"title": "Client Name", "value": entry.fields_by_name('Full Name')[0].value},
        ]

        # 2. Construct the Adaptive Card JSON
        adaptive_card = {
            "type": "message",
            "attachments": [
                {
                    "contentType": "application/vnd.microsoft.card.adaptive",
                    "content": {
                        "$schema": "http://adaptivecards.io/schemas/adaptive-card.json",
                        "type": "AdaptiveCard",
                        "version": "1.2",
                        "body": [
                            {
                                "type": "TextBlock",
                                "text": form_title,
                                "weight": "Bolder",
                                "size": "Medium",
                                "wrap": True
                            },
                            {
                                "type": "FactSet",
                                "facts": facts
                            }
                        ]
                    }
                }
            ]
        }
        return adaptive_card

    def send_to_teams(self):
        try:
            # Verify request signature
            if self._au.verify_request(self._event['headers'].get('authorization', '')):

                # Parse the webhook data
                self._entry = FormEntry.create(json.loads(self._event['body']))

                # Check priority: Only send urgent/critical requests
                priority_field = self._entry.fields_by_name('Priority')

                if priority_field and priority_field[0].value in ('Urgent', 'Critical'):
                    try:
                        http = urllib3.PoolManager()
                        resp = http.request(
                            'POST',
                            os.environ['TEAMS_WEBHOOK_URL'],
                            body=json.dumps(self.create_adaptive_card()),
                            headers={'Content-Type': 'application/json'}
                        )

                        if resp.status not in (200, 201, 202):
                            logger.warning(f"Teams message delivery failed: {resp.status} {resp.data}")
                        else:
                            logger.info(f"Message for form '{self._entry.form_name}' sent to Teams.")

                    except urllib3.exceptions.HTTPError as err:
                        logger.warning(f"HTTP Error: {err}")
                else:
                    logger.info("Submission not urgent; skipping Teams notification.")

                return {
                    'statusCode': 200,
                    'body': json.dumps('Processed')
                }
            else:
                return {
                    'statusCode': 401,
                    'body': json.dumps('Invalid webhook signature.')
                }

        except AuthenticationError as err:
            logger.warning(f"Auth Error: {err}")
            return {
                'statusCode': 401,
                'body': json.dumps('Authentication failed.')
            }
        except Exception as e:
            logger.error(f"Unexpected error: {e}")
            return {
                'statusCode': 500,
                'body': json.dumps('Internal Server Error')
            }

def lambda_handler(event, context):
    return TeamsSender(event).send_to_teams()
