# -*- coding: utf-8 -*-

"""
FormSmarts API & Webhook Client version 2 for Python 3 -- v.2.0-2025-06-16
https://formsmarts.com/api-webhook-client

Dependency: PyJWT (https://pyjwt.readthedocs.io/en/stable/)
You can install PyJWT with "pip": pip install pyjwt
Works with Python >= v. 3.8. Contact Support https://f8s.co/support if you need to use an earlier version of Python.

"""

import time
import datetime
import hashlib
import secrets
import urllib3
import json
import decimal
import jwt
from urllib.parse import urljoin


API_HOST = 'https://formsmarts.com'


class Authenticator:
    
    ISSUER = 'formsmarts.com'
    ALGORITHMS = ['HS256', 'HS384', 'HS512']
    CLOCK_SKEW = 5  # seconds
    AUTH_HEADER_SCHEME = 'Bearer'
    AUTH_HEADER_SCHEME_LC = AUTH_HEADER_SCHEME.lower()

    @classmethod
    def parse_authorization_header(cls, header):
        """
        Parse bearer token from the Authorization header.
        :param header: header value as a str
        :return: a bearer token
        """
        try:
            scheme, token = header.split(' ')
        except ValueError:
            raise AuthenticationError('Authorization header format is invalid.')
        if scheme.lower() != cls.AUTH_HEADER_SCHEME_LC:
            raise AuthenticationError('Authorization header scheme is invalid.')
        return token

    @classmethod
    def parse_account_id(cls, token):
        """
        Parse the client's unverified Account ID from the JWT bearer token.
        :param token: bearer token from the Authorization header
        :return: an unverified Account ID
        """
        try:
            return jwt.decode(token, options={"verify_signature": False})['sub']
        except jwt.InvalidTokenError as err:
            raise AuthenticationError(str(err))
        except KeyError:
            raise AuthenticationError('Account ID missing from the Authorization header.')

    def verify_request(self, authorization_header):
        raise NotImplementedError('Please use a subclass.')


class WebhookAuthenticator(Authenticator):
    """
    This class allows FormSmarts users to verify a webhook callback originates from FormSmarts
    before processing the JSON data received in the request body.
    https://formsmarts.com/online-form-api-webhook#message-authentication
    """

    AUDIENCE = 'webhook'

    def __init__(self, webhook_key):
        """
        :param webhook_key: Your FormSmarts Webhook Key (https://formsmarts.com/account/view#security-settings)
        """
        self._wh_key = webhook_key

    def verify_request(self, authorization_header):
        """
        Verify that the webhook message is authenticated as originating from FormSmarts. Raises an AuthenticationError
        if verification fails.
        :param authorization_header: the Authorization header of the HTTP request
        :raise AuthenticationError: if the Authorization header is missing or unparsable or the request has expired or
        the security token is invalid.
        """
        token = self.parse_authorization_header(authorization_header)
        try:
            return jwt.decode(
                token,
                self._wh_key,
                issuer=self.ISSUER,
                audience=self.AUDIENCE,
                leeway=self.CLOCK_SKEW,
                algorithms=self.ALGORITHMS,
                options=dict(
                    require=['iss', 'sub', 'aud', 'iat', 'nbf', 'exp'],
                    verify_iss=True,
                    verify_aud=True,
                    verify_iat=True,
                    verify_nbf=True,
                    verify_exp=True,
                )
            )
        except jwt.InvalidTokenError as err:
            raise AuthenticationError(str(err))


class APIAuthenticator(Authenticator):
    """
    This class allows users to authenticate requests to FormSmarts API.
    https://formsmarts.com/online-form-api-webhook#message-authentication
    """

    AUDIENCE = 'api'
    ALGORITHM = 'HS256'
    TTL = 5 * 60  # Token expires after 5 min
    MIN_TTL = 15  # Token may be reused if it is still for valid more than 15 sec
    AUTH_HEADER = 'Authorization'
    AUTH_HEADER_SCHEME = 'Bearer'

    def __init__(self, account_id, api_key):
        """
        :param account_id: Your FormSmarts Account ID (https://formsmarts.com/account/view)
        :param api_key: Your FormSmarts Webhook Key (https://formsmarts.com/account/view#security-settings)
        """
        self.validate_account_id(account_id)
        self._acc_id = account_id
        self._api_key = api_key
        self._token = None

    def get_access_token(self):
        """
        Return a token suitable for accessing the FormSmarts API.
        """
        if not self._token or self._token['expires'] - time.time() < self.MIN_TTL:
            self.create_token()
        return self._token['value']

    def get_single_use_token(self):
        """
        Return a single-use FormSmarts API token.
        """
        return self.create_token(use_nonce=True)['value']

    def create_token(self, use_nonce=False):
        """
        :param use_nonce: if True, a nonce is added to prevent the token from being reused
        :return: a JWT token
        """
        now = int(time.time())
        expires = now + self.TTL
        payload = dict(
            iss=self.ISSUER,
            sub=self._acc_id,
            aud=self.AUDIENCE,
            iat=now,
            nbf=now,
            exp=expires,
        )
        if use_nonce:
            payload['nonce'] = self.create_nonce()
        tok = jwt.encode(
            payload,
            self._api_key,
            algorithm=self.ALGORITHM,
        )
        self._token = dict(expires=expires, value=tok)
        return self._token

    def get_authorization_header(self):
        return ' '.join((self.AUTH_HEADER_SCHEME, self.get_access_token()))

    @staticmethod
    def create_nonce():
        return secrets.token_urlsafe()

    @staticmethod
    def validate_account_id(acc_id):
        try:
            if acc_id.startswith('FSA-') and acc_id[4:].isdigit():
                return True
        except (TypeError, AttributeError):
            pass
        raise AuthenticationError('Invalid Account ID: {}.'.format(acc_id))


class APIError(RuntimeError):
    def __init__(self, message):
        super().__init__(message)


class AuthenticationError(APIError):
    def __init__(self, message):
        super().__init__(message)


class APIRequestError(APIError):
    def __init__(self, message, status, text=None, json=None):
        super().__init__(message)
        self._status = status
        self._text = text
        self._json = json

    @property
    def status(self):
        """ The HTTP response status """
        return self._status

    @property
    def text(self):
        """ The HTTP response text """
        return self._text

    @property
    def json(self):
        """ The HTTP response JSON payload """
        return self._json


class Field:
    """
    An input field.
    """

    REVISION_URL = urljoin(API_HOST, '/api/v1/entries/{}/fields/{}/revisions')

    def __init__(self, id, type, name, value):
        self._id = id
        self._type = type
        self._name = name
        self._val = value
        self._entry = None

    @property
    def id(self):
        """ The field's ID """
        return self._id

    @property
    def name(self):
        """ The field's name """
        return self._name

    @property
    def type(self):
        """ The field's type """
        return self._type

    @property
    def value(self):
        """ The field's value """
        return self._val

    @value.setter
    def value(self, new_value):
        """
        Amend the field's value.
        https://formsmarts.com/weblog/collaboration/how-to-edit-submissions-made-on-your-forms
        """
        hsh = hashlib.sha1(usedforsecurity=False)
        hsh.update(str(self._val).encode())
        resp = request(
            'POST',
            self.REVISION_URL.format(self.entry.reference_number, self.id),
            fields=dict(
                new_value=str(new_value),
                hash=hsh.hexdigest()
            ),
            headers=self.entry.headers
        )
        # Assignment Expressions only work with Python v. >= 3.8.
        if (code := resp.json().get('code')) and code == 'ok':
            self._val = new_value

    @property
    def entry(self):
        return self._entry

    def set_entry(self, entry):
        """
        :param entry: a FormEntry object
        """
        self._entry = entry


class Date(Field):
    """
    A date field.
    """

    def __init__(self, id, type, name, value):
        super().__init__(id, type, name, value)
        self._dt_val = None

    @property
    def value(self):
        """ The field's value """
        if not self._val:
            return self._val
        if not self._dt_val:
            # Webhook dates are UTC without timezone
            # TODO: clarify if (1) we do anything to change the value entered by the user on the form
            # TODO: and (2) we there is any convertion to UTC done by the browser
            # val = datetime.datetime.fromisoformat('{}+00:00'.format(val))
            self._dt_val = datetime.date.fromisoformat(self._val)
        return self._dt_val

    @value.setter
    def value(self, new_value):
        """ Edit the field. """
        super().value(new_value)
        self._dt_val = None


class Upload(Field):
    """
    A file upload field.
    """

    DOWN_URL = urljoin(API_HOST, '/api/v1/attachments/{}')
    UPL_URL = urljoin(API_HOST, '/api/v1/attachments')
    ACCESS_URL = 'https://formsmarts.com/attachments/{}'

    def __init__(self, id, type, name, value):
        super().__init__(id, type, name, value)
        try:
            self._fn = value.get('filename')
            self._upl_id = value.get('attachment_id')
            self._ps_url = value.get('presigned_url')
        except AttributeError:
            # 'value' is a dict() in the API response, but a str in the webhook message
            self._fn = None
            self._upl_id = None
            self._ps_url = None
        self._url = None
        self._io_buf = None

    @classmethod
    def upload(cls, api_authenticator, form_id, field_id, io_stream, filename, content_type):
        """
        Upload a form attachment.
        https://formsmarts.com/form-response-api
        :param api_authenticator: an APIAuthenticator object
        :param field_id: a field ID
        :param form_id: a form ID (if a form's URL is https://f8s.co/xyz, its ID is "xyz")
        :param io_stream: an I/O stream object
        :param filename: name of the new file
        :param content_type: the attachment's MIME Content Type, e.g. 'image/png'
        :return: an Upload ID
        """
        with io_stream:
            doc = io_stream.read()
            data = {
                'form_id': form_id,
                str(field_id): (filename, doc, content_type),
            }
            resp = request(
                'POST',
                cls.UPL_URL,
                fields=data,
                headers={APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()}
            )
            return resp.json()['files'][0]['id']

    @property
    def value(self):
        """
        Use download() to download an attachment into a bytes stream object.
        """
        return None

    @value.setter
    def value(self, new_value):
        """
        Use replace() to replace a form attachment with a new one.
        """
        pass

    def download(self, io_stream):
        """
        Download the attachment into an I/O stream object.
        :param io_stream: an I/O stream object.
        """
        if not (uid := self.upload_id):
            raise APIError('Upload field is empty, nothing to download.')
        resp = request(
            'GET',
            self.DOWN_URL.format(uid),
            headers=self.entry.headers
        )
        with io_stream:
            io_stream.write(resp.data)

    def replace(self, io_stream, filename, content_type):
        """
        Replace the attachment with a new one.
        :param io_stream: an I/O stream object
        :param filename: name of the new file
        :param content_type: the attachment's MIME Content Type, e.g. 'image/png'
        """
        with io_stream:
            doc = io_stream.read()
            data = {
                'form_id': self.entry.form_id,
                str(self._id): (filename, doc, content_type),
            }
            resp = request(
                'POST',
                self.UPL_URL,
                fields=data,
                headers=self.entry.headers
            )
            upload_id = resp.json()['files'][0]['id']
            self._revise_field(upload_id, filename, doc)

    def _revise_field(self, upload_id, filename, document):
        val = ':'.join((upload_id, filename))
        hsh = hashlib.sha1(usedforsecurity=False)
        uid = self.upload_id
        cur_val = ':'.join((self._upl_id, self._fn)) if uid else ''
        hsh.update(cur_val.encode())
        f_hsh = hashlib.sha256()
        f_hsh.update(document)
        resp = request(
            'POST',
            self.REVISION_URL.format(self.entry.reference_number, self.id),
            fields=dict(
                new_value=val,
                hash=hsh.hexdigest(),
                file_hash=f_hsh.hexdigest(),
            ),
            headers=self.entry.headers
        )
        # Assignment Expressions only work with Python v. >= 3.8.
        if (code := resp.json().get('code')) and code == 'ok':
            self.upload_id = upload_id
            self.filename = filename

    @property
    def filename(self):
        return self._fn

    @filename.setter
    def filename(self, filename):
        self._fn = filename

    @property
    def api_url(self):
        if uid := self.upload_id:
            return self.DOWN_URL.format(uid)

    @property
    def upload_id(self):
        if not self._upl_id:
            # Object must have been created from a webhook
            if self._url:
                self._upl_id = self._url.split('/')[-1]
        return self._upl_id

    @upload_id.setter
    def upload_id(self, upload_id):
        self._upl_id = upload_id

    @property
    def url(self):
        if not self._url:
            # Only set when object is created from a webhook
            if self._upl_id:
                self._url = self.ACCESS_URL.format(self._upl_id)
        return self._url

    @url.setter
    def url(self, url):
        if self._url is not None:
            raise APIError('URL was already set.')
        self._url = url
    @property
    def presigned_url(self):
        return self._ps_url

    @presigned_url.setter
    def presigned_url(self, ps_url):
        if self._ps_url is not None:
            raise APIError('URL was already set.')
        self._ps_url = ps_url


class Signature(Upload):
    """
    A signature field.
    """

    def __init__(self, id, type, name, value):
        super().__init__(id, type, name, value)
        try:
            self._val = value.get('text')
        except AttributeError:
            # 'value' is a str in the webhook message
            self._val = None

    def replace(self, io_stream, filename, content_type):
        raise NotImplementedError("Signatures can't be updated.")


class FormEntry:
    """
    A form entry.
    """

    FIELD_SUBCLASSES = {'upload': Upload, 'signature': Signature, 'date': Date}
    TAG_URL = urljoin(API_HOST, '/api/v1/entries/{}/tags')
    SHARE_URL = urljoin(API_HOST, '/api/v1/entries/{}/share')
    SEARCH_BY_DATES_URL = urljoin(API_HOST, '/api/v1/forms/{}/entries/by/dates')
    SEARCH_URL = urljoin(API_HOST, '/api/v1/entries/search')
    BATCH_URL = urljoin(API_HOST, '/api/v1/forms/{}/entries/batch')
    API_RESP_LIMIT = 250
    SEARCH_RESP_LIMIT = 100
    TRUE_VALS = ('yes', 'true')

    def __init__(self):
        self._id = None
        self._ref_num = None
        self._date = None
        self._name = None
        self._au = None
        self._fields = []
        self._id_fields = {}
        self._type_fields = None
        self._name_fields = None
        self._ver = None
        self._ctx = None
        self._pay = None
        self._amt_due = None
        self._tags = None
        self._sys_tags = None

    @classmethod
    def create(cls, message, api_authenticator=None, structure=None):
        """
        Create a FormEntry object from a webhook callback message or the response of a Form Entry API request.
        :param message: a webhook message or Form Entry API response
        :param api_authenticator: an APIAuthenticator object
        :param structure: the structure of the form like the one returned by APIFormEntry().form_structure
        :return: a FormEntry object
        """
        if message.get('api_id') == 'FSEVAPI':
            return WebhookEntry(message)
        else:
            if not api_authenticator:
                raise APIError('An APIAuthenticator object is required.')
            e = APIEntry(message, structure)
            if api_authenticator:
                e.set_authenticator(api_authenticator)
            return e

    @classmethod
    def search(cls, api_authenticator, query, form_id=None, tags=None, system_tags=None):
        """
        Search form entries by email address, phone number the 'latest' keyword, or any ID associated with entry:
        https://formsmarts.com/weblog/online-form/explore-form-results-with-data-search
        Limitation: Draft form entries are ignored: https://formsmarts.com/save-a-form-and-continue-later
        :param api_authenticator: an APIAuthenticator object
        :param query: an email address, phone number, ID or the 'latest' keyword as a str object
        :param form_id: a form ID (if a form's URL is https://f8s.co/xyz, its ID is "xyz")
        :param tags: a tag, see: https://formsmarts.com/view-form-response-online#monitor
        :param system_tags: a system tag, see: https://formsmarts.com/view-form-response-online#system-tags
        :return: a generator yielding APIEntry objects
        """
        offset = 0
        has_more = True
        structs = {}  # Forms structure
        params = dict(query=query, limit=cls.SEARCH_RESP_LIMIT)
        for key in ('form_id', 'tags', 'system_tags'):
            if val := locals().get(key):
                params[key] = val if isinstance(val, str) else ','.join(val)  # Tags are lists or tuples
        while has_more:
            params['offset'] = offset
            headers = {APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()}
            search_resp = request('GET', cls.SEARCH_URL, fields=params, headers=headers).json()
            n_results = len(search_resp)
            offset += n_results
            has_more = n_results == cls.SEARCH_RESP_LIMIT
            entries_by_form = {}
            for fid, tr_ids in cls._group_tr_ids_by_form(search_resp, form_id=form_id).items():
                batch = cls.fetch_batch(api_authenticator, fid, tr_ids, raw=True)
                entries_by_form[fid] = {e['metadata']['reference_number']: e for e in batch['entries']}
                structs[fid] = batch['fields']
            for res in search_resp:
                if 'reference_number' not in res:
                    continue  # Draft form entries aren't supported yet
                fid = form_id or res['form_id']
                yield APIEntry(
                    {'entries': [entries_by_form[fid][res['reference_number']]], 'form': {'id': fid}},
                    structure= {'fields': structs[fid], 'form': {'id': fid}}
                )

    @classmethod
    def search_by_dates(cls, api_authenticator, form_id, start_date, end_date=None, timezone=None, structure=None):
        """
        Return form entries submitted between two dates as a generator.
        :param api_authenticator: an APIAuthenticator object
        :param form_id: a form ID (if a form's URL is https://f8s.co/xyz, its ID is "xyz")
        :param start_date: start date as a date object or ISO string YYYY-MM-DD
        :param end_date: optional end date as a date object or ISO string YYYY-MM-DD
        :param timezone: optional timezone used for dates in the API response, for example "Europe/London".
        :param structure: the structure of the form like the one returned by APIFormEntry().form_structure
        :return: a generator yielding APIEntry objects
        """
        offset = 0
        has_more = True
        last_req_tm = 0
        struct = structure
        params = dict(start_date=get_iso_date(start_date), limit=cls.API_RESP_LIMIT)
        url = cls.SEARCH_BY_DATES_URL.format(form_id)
        if not struct:
            params['structure'] = 'true'
        if end_date:
            params['end_date'] = get_iso_date(end_date)
        if timezone:
            params['timezone'] = timezone
        while has_more:
            params['offset'] = offset
            if struct and 'structure' in params:
                params.pop('structure')
            headers = {APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()}
            resp = request('GET', url, fields=params, headers=headers).json()
            n_results = len(resp['entries'])
            offset += n_results
            has_more = n_results == cls.API_RESP_LIMIT
            # Set the 'form' property that the Form by Dates API endpoint doesn't include in the response
            resp['form'] = {'id': form_id}
            if not struct:
                struct = {'fields': resp['fields'], 'form': {'id': form_id}}
            for i in range(n_results):
                yield APIEntry(resp, structure=struct, message_index=i)

    @classmethod
    def fetch_batch(cls, api_authenticator, form_id, reference_numbers, timezone=None, structure=None, raw=False):
        """
        Return the form entries with the reference numbers provided as a generator.
        :param api_authenticator: an APIAuthenticator object
        :param form_id: a form ID (if a form's URL is https://f8s.co/xyz, its ID is "xyz")
        :param reference_numbers: a list of Reference Numbers
        :param timezone: optional timezone used for dates in the API response, for example "Europe/London".
        :param structure: the structure of the form like the one returned by APIFormEntry().form_structure
        :param raw: return the raw API response instead of a generator of APIEntry objects
        :return: a generator yielding APIEntry objects or a raw API response
        """
        url = cls.BATCH_URL.format(form_id)
        params = dict(reference_numbers=reference_numbers)
        params['structure'] = not structure
        params['timezone'] = timezone or None
        headers = {APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()}
        resp = request('POST', url, json=params, headers=headers).json()
        if raw:
            return resp
        return cls._create_results_generator(resp, form_id, structure)

    @property
    def fields(self):
        """
        Return a list of fields in the order they appear on the form.
        """
        if not self._fields:
            self._make_fields()
        return self._fields

    @property
    def form_id(self):
        return self._id

    @property
    def reference_number(self):
        return self._ref_num

    @property
    def date_submitted(self):
        return self._date

    @property
    def form_name(self):
        return self._name

    @property
    def api_version(self):
        return self._ver

    @property
    def context(self):
        """
        Form submission context.
        https://formsmarts.com/weblog/form-builder/track-source-of-leads-web-form
        :return: a dict {'label': '', 'value': ''}
        """
        return self._ctx

    @property
    def payment(self):
        """
        Payment associated with this form.
        :return: a dict {'amount': Decimal('0.00'), 'currency': 'USD', 'transaction_id': '', processor_name=''}
        """
        return self._pay

    def field_by_id(self, field_id):
        """
        Look up a field by Field ID.
        :param field_id: the Field's ID
        :return: a field property
        :raise KeyError: if the Field ID doesn't exist
        """
        if not self._id_fields:
            self._make_fields()
        return self._id_fields[field_id]

    def fields_by_type(self, field_type):
        if not self._type_fields:
            t_fields = {}
            for f in self.fields:
                if f.type not in t_fields:
                    t_fields[f.type] = []
                t_fields[f.type].append(f)
            self._type_fields = t_fields
        try:
            return self._type_fields[field_type.lower()]
        except KeyError:
            return []

    def fields_by_name(self, field_name):
        fn = field_name.strip().lower()
        if not self._name_fields:
            n_fields = {}
            for f in self.fields:
                nm = f.name.lower()
                if nm not in n_fields:
                    n_fields[nm] = []
                n_fields[nm].append(f)
            self._name_fields = n_fields
        try:
            return self._name_fields[fn]
        except KeyError:
            return []

    @property
    def tags(self):
        """
        Get tags associated with this form entry.
        https://formsmarts.com/view-form-response-online#tag
        """
        if self._tags is None:
            resp = request('GET', self.TAG_URL.format(self._ref_num), headers=self.headers).json()
            self._tags = resp['user_tags']
            self._sys_tags = resp['system_tags']
        return self._tags

    @property
    def system_tags(self):
        """
        Get tags assigned to this form entry by FormSmarts.
        https://formsmarts.com/view-form-response-online#system-tags
        """
        if self._sys_tags is None:
            self.tags()
        return self._sys_tags

    def add_tag(self, tag):
        if self._tags and tag in self._sys_tags:
            return
        request('POST', self.TAG_URL.format(self._ref_num), fields=dict(tag=tag), headers=self.headers)
        if self._tags is not None and not tag in self._tags:
            self._tags.append(tag)

    def share(self, emails):
        """
        Share the form entry by email.
        https://formsmarts.com/view-form-response-online#share
        :param emails: a list of email addresses
        """
        request('POST', self.SHARE_URL.format(self._ref_num), json=dict(emails=emails), headers=self.headers)

    def _to_py_type(self, field_type, val):
        if val == '' and field_type in ('number', 'posint', 'boolean'):
            return None
        match field_type:
            case 'number':
                return decimal.Decimal(val)
            case 'posint':
                return int(val)
            case 'boolean':
                return val in self.TRUE_VALS
            case _:
                return val

    def _make_fields(self):
        return NotImplementedError('Please use a subclass.')

    @staticmethod
    def _create_results_generator(api_resp, form_id, structure):
        """
        :param api_resp: the JSON response of an API endpoint returning multiple entries
        :param form_id: a Form ID
        :param structure: structure of the form
        :return: a generator yielding APIEntry objects
        """
        # Set the 'form' property that the 'Form by Dates' and 'Batch Entries' API endpoints doesn't include
        # in the response
        api_resp['form'] = {'id': form_id}
        if not structure:
            structure = {'fields': api_resp['fields'], 'form': {'id': form_id}}
        for i in range(len(api_resp['entries'])):
            yield APIEntry(api_resp, structure=structure, message_index=i)

    @staticmethod
    def _group_tr_ids_by_form(search_response, form_id=None):
        """
        Group Transaction IDs by form.
        :param search_response: Search API results
        :param form_id: a Form ID for searches limited to a specific form
        :return: a dict object
        """
        tr_ids_by_form = {}
        for res in search_response:
            fid = form_id or res['form_id']
            if fid not in tr_ids_by_form:
                tr_ids_by_form[fid] = []
            try:
                tr_ids_by_form[fid].append(res['reference_number'])
            except KeyError:
                pass  # The API Client doesn't support Draft form entries (which don't have a Reference Number)
        return tr_ids_by_form

    def _set_optional_entry_attributes(self, msg_entry):
        """
        Set context, payment and amount due attributes, if applicable
        :param msg_entry: a webhook request or single entry from an API response
        """
        self._ctx = msg_entry.get('context')
        self._pay = msg_entry.get('payment')
        if self._pay:
            to_decimal_amount(self._pay)
        if 'amount_due' in msg_entry:
            self._amt_due = msg_entry.get('amount_due')
            to_decimal_amount(self._amt_due)

    def set_authenticator(self, api_authenticator):
        """
        :param api_authenticator: an APIAuthenticator object
        """
        self._au = api_authenticator

    @property
    def authorization_header(self):
        if not self._au:
            raise APIError("API Authenticator isn't set.")
        return self._au.get_authorization_header()

    @property
    def headers(self):
        return {APIAuthenticator.AUTH_HEADER: self.authorization_header}

    @property
    def notes(self):
        raise NotImplementedError('Not implemented yet.')

    def add_note(self, note):
        raise NotImplementedError('Not implemented yet.')

    def delete(self):
        """
        Permanently delete this form entry.
        <!> The form entry is deleted immediately, there is no way to recover the data.
        """
        raise NotImplementedError('Not implemented yet.')


class WebhookEntry(FormEntry):

    def __init__(self, message):
        super().__init__()
        self._wh_msg = message
        self._id = message['form_id']
        self._ref_num = message['fs_ref_num']
        self._name = message['form_name']
        self._date = datetime.datetime.fromisoformat('{}Z'.format(message['api_date']))
        self._ver = int(message['api_version'])
        self._set_optional_entry_attributes(message)

    @property
    def form_name(self):
        return self._name

    @property
    def amount_due(self):
        """
        Amount due with this form.
        https://formsmarts.com/weblog/payment-form/online-form-with-check-or-cash-payment
        :return: a dict {'amount': Decimal('0.00'), 'currency': 'USD'}
        """
        return self._amt_due

    @property
    def is_validation_hook(self):
        """
        Return True if this object was created from a validation webhook.
        https://formsmarts.com/form-validation-webhook
        """
        return self._wh_msg.get('action') == 'validate'

    def _make_fields(self):
        fields = self._wh_msg['fields']
        for fd in fields:
            tp = fd['field_datatype']
            is_sig = tp == 'signature'
            is_upl = tp == 'attachment'
            val = fd['field_value']
            if is_upl:
                tp = 'upload'
            val = self._to_py_type(tp, val)
            cls = self.FIELD_SUBCLASSES.get(tp, Field)
            f = cls(fd['field_id'], tp, fd['field_name'], val)
            if is_upl:
                f.filename = fd.get('attachment_filename')
            if is_sig or is_upl:
                f.url = fd.get('attachment_url')
                f.presigned_url = fd.get('attachment_presigned_url')
            f.set_entry(self)
            self._id_fields[f.id] = f
            self._fields.append(f)


class APIEntry(FormEntry):

    ENTRY_URL = urljoin(API_HOST, '/api/v1/entries/{}')
    SUBMIT_URL = urljoin(API_HOST, '/api/v1/forms/{}/entries')
    PDF_URL = urljoin(API_HOST, '/api/v1/entries/{}.pdf')

    def __init__(self, message, structure=None, message_index=0):
        super().__init__()
        self._msg = None
        self._msg_idx = message_index
        self._ip = None
        self._msg = message
        self._struct = None
        self._load(structure)

    @classmethod
    def fetch(cls, api_authenticator, ref_num, structure=None, timezone=None):
        """
        Fetch a form entry from its Reference Number.
        https://formsmarts.com/form-response-api
        :param api_authenticator: an APIAuthenticator object
        :param ref_num: the Reference Number of a form entry
        :param structure: the structure of the form returned by APIFormEntry().form_structure()
        :param timezone: Timezone used for dates in the API response, for example "Europe/London".
        :return: the response of a Form Entry API request
        """
        params = dict(structure=json.dumps(not bool(structure)))
        if timezone:
            params['timezone'] = timezone
        resp = request(
            'GET',
            cls.ENTRY_URL.format(ref_num.upper()),
            headers={APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()},
            fields=params
        )
        return cls.create(resp.json(), api_authenticator=api_authenticator, structure=structure)

    @classmethod
    def submit(cls, api_authenticator, form_id, field_values, context=None):
        """
        Submit a form.
        https://formsmarts.com/form-submission-api
        :param api_authenticator: an APIAuthenticator object
        :param form_id: a form ID (if a form's URL is https://f8s.co/xyz, its ID is "xyz")
        :param field_values: a dict mapping field IDs to their value
        :param context: a form context parameter
        :return: a FormSmarts Reference Number
        """
        data = {str(k): str(v) for k, v in field_values.items()}
        if context:
            data['form_context'] = context
        resp = request(
            'POST',
            cls.SUBMIT_URL.format(form_id.lower()),
            headers={APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()},
            fields=data
        )
        return resp.json()['reference_number']

    @classmethod
    def download_pdf(cls, api_authenticator, ref_num, io_stream, timezone=None):
        """
        Download the form entry as a PDF document
        https://formsmarts.com/form-response-api#pdf-api
        :param api_authenticator: an APIAuthenticator object
        :param ref_num: the Reference Number of a form entry
        :param io_stream: a binary stream
        :param timezone: Timezone used for dates in the API response, for example "Europe/London".
        """
        params = {'timezone': timezone} if timezone else {}
        resp = request(
            'GET',
            cls.PDF_URL.format(ref_num),
            headers={APIAuthenticator.AUTH_HEADER: api_authenticator.get_authorization_header()},
            fields=params
        )
        with io_stream:
            io_stream.write(resp.data)

    def entries_by_dates(self, start_date, end_date=None, timezone=None):
        """
        Return form entries submitted between two dates.
        :param start_date: start date as a date object or ISO string YYYY-MM-DD
        :param end_date: optional end date as a date object or ISO string YYYY-MM-DD
        :param timezone: optional timezone used for dates in the API response, for example "Europe/London".
        :return: a generator yielding FormEntry objects
        """
        return self.search_by_dates(
            api_authenticator=self._au,
            form_id=self.form_id,
            start_date=start_date,
            end_date=end_date,
            timezone=timezone,
            structure=self.form_structure,
        )

    @property
    def ip_address(self):
        return self._ip

    def _load(self, structure):
        msg = self._msg
        self._id = msg['form']['id']
        self._set_structure(structure)
        md = msg['entries'][self._msg_idx]['metadata']
        self._ref_num = md['reference_number']
        self._date = datetime.datetime.fromisoformat(md['date_submitted'])
        self._ip = md.get('ip_address')
        self._set_optional_entry_attributes(msg['entries'][self._msg_idx])

    def _make_fields(self):
        fields = self._struct['fields']
        values = self._msg['entries'][self._msg_idx]['entry']
        for fd, val in zip(fields, values):
            tp = fd['type']
            cls = self.FIELD_SUBCLASSES.get(tp, Field)
            val = self._to_py_type(tp, val)
            f = cls(fd['id'], tp, fd['name'], val)
            f.set_entry(self)
            self._id_fields[f.id] = f
            self._fields.append(f)

    @property
    def form_structure(self):
        return self._struct

    def _set_structure(self, structure):
        if not structure:
            self._struct = {
                'form': self._msg['form'],
                'fields': self._msg['fields']
            }
        else:
            try:
                if (not self._id or structure['form']['id'] == self._id) and structure['fields'] is not None:
                    self._struct = structure
                else:
                    raise APIError("Did you pass the structure of the wrong form?")
            except KeyError:
                raise APIError('Not a valid form structure.')


IS_URLLIB3_V1 = int(urllib3.__version__.split('.')[0]) < 2

class Request:

    DELAY_BTW_REQS = 0.1  # 100 ms

    def __init__(self):
        self._http = urllib3.PoolManager()
        self._last_req_tm = 0

    def __call__(self, method, url, **kwargs):
        """
        Submit a request
        :param method: an HTTP method
        :param url: a URL
        :param kwargs: valid keys include: headers, body, json
        """
        if (elapsed := time.time() - self._last_req_tm) < self.DELAY_BTW_REQS:
            time.sleep(self.DELAY_BTW_REQS - elapsed)
        self._last_req_tm = time.time()
        return self.do_request(method, url, **kwargs)

    def do_request(self, method, url, **kwargs):
        if 'json' in kwargs and IS_URLLIB3_V1:
            kwargs['json'] = json.dumps(kwargs['json']).encode()
        resp = self._http.request(method, url, **kwargs)
        if resp.status == 200:
            return resp
        try:
            json_resp = resp.json()
            txt_resp = json_resp['message'] if 'message' in json_resp else str(json_resp)
        except json.decoder.JSONDecodeError:
            json_resp = None
            txt_resp = resp.data.decode()
        raise APIRequestError(
            "Request to {} returned status code: {} with the message: {}".format(
                url, resp.status, txt_resp
            ),
            status=resp.status,
            text=txt_resp,
            json=json_resp,
        )

request = Request()


def to_decimal_amount(a_dict):
    if amt := a_dict.get('amount'):
        a_dict['amount'] = decimal.Decimal(amt)

def get_iso_date(a_date):
    """
    :param a_date: a datetime or date object or an ISO date string
    :return: an ISO date string
    """
    if isinstance(a_date, datetime.datetime):
        return a_date.date().isoformat()
    else:
        return a_date
